using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Runtime.InteropServices;
using System.Text;
using System.Windows.Forms;
using DynaPDF;
using ArrayList = System.Collections.ArrayList;

namespace PDFViewer
{
   public partial class Form1 : Form
   {
      [DllImport("gdi32.dll", CharSet = CharSet.Unicode)] private static extern int GetICMProfileW(IntPtr hDC, ref int Len, StringBuilder Filename);

      public Form1()
      {
         InitializeComponent();
      }

      private void Form1_Load(object sender, EventArgs e)
      {
         String path;
         m_Cache     = PDFCanvas.GetCacheInstance();
         m_PDF       = PDFCanvas.GetPDFInstance();
         m_OpenFlags = TInitCacheFlags.icfIgnoreOpenAction | TInitCacheFlags.icfIgnorePageLayout;
         if (PDFCanvas.ColorManagement)
         {
            // Because .Net applications require an installer before they can be executed outside of Visual Studio, we never know
            // where the app will be installed, and hence, we cannot access the directory /examples/test_files/. However, it should
            // at least be possible to activate color management in the demo app. To archive this, the default CMYK profile is embedded
            // in the assembly. This is of course not what you should do in a real application because CMYK profiles are rather large
            // and we need a lot of additional files, e.g. the CMap files. If you work with C# then you should have a license of a
            // professional installer so that you don't need to embed such files into the assembly.

            // Very important: All profile paths must be absolute paths! This is required because the open file dialog changes the
            // current directory and if DynaPDF must reload a profile later then it is maybe no longer possible since the current
            // directory was changed.
            System.IO.FileStream file = null;
            path = System.IO.Path.GetFullPath("deficcprofile.icm");
            if (!System.IO.File.Exists(path))
            {
               try
               {
                  // Extract the CMYK profile from the assembly and store it in the application folder.
                  System.Reflection.Assembly a = System.Reflection.Assembly.GetExecutingAssembly();
                  System.IO.Stream strm = a.GetManifestResourceStream("PDFViewer.Resources.ISOcoated_v2_bas.ICC");
                  file = new System.IO.FileStream("deficcprofile.icm", System.IO.FileMode.CreateNew);

                  int len = 4096;
                  Byte[] buffer = new Byte[len];
                  int bytesRead = strm.Read(buffer, 0, len);
                  // write the required bytes
                  while (bytesRead > 0)
                  {
                     file.Write(buffer, 0, bytesRead);
                     bytesRead = strm.Read(buffer, 0, len);
                  }
                  strm.Close();
                  file.Close();
               }catch(Exception ex)
               {
                  if (file != null) file.Close();
                  PDFCanvas.AddError(ex.Message);
               }
            }
            // Gray and RGB profiles are automatically created if not provided. The default RGB profile is sRGB.
            // It is possible to set all profiles just to null. In this case, CMYK colors will only be rendered with
            // color management if the PDF file contains ICCBased color spaces or a CMYK Output Intent.
            // If you want to disable color management (if it was enabled) then set the parameter Profiles to null, e.g.
            // CPDF.rasInitColorManagement(m_Cache, null, TPDFInitCMFlags.icmBPCompensation);
            TPDFColorProfiles profiles = new TPDFColorProfiles();
            int size = 0;
            // This is the private dc of the PDF Canvas. It is not required to release this dc.
            IntPtr dc = PDFCanvas.GetDC();
            StringBuilder iccFile = new StringBuilder();
            // Get the monitor profile
            GetICMProfileW(dc, ref size, null);
            if (size > 0)
            {
               iccFile.Length = size * 2 + 2; // 2 bytes per character plus 2 bytes for the null-terminator.
               GetICMProfileW(dc, ref size, iccFile);
               profiles.DeviceProfileW = iccFile.ToString();
            }
            profiles.StructSize = Marshal.SizeOf(profiles);
            profiles.DefInCMYKW = path;
            if (CPDF.rasInitColorManagement(m_Cache, profiles, TPDFInitCMFlags.icmBPCompensation) == 0)
            {
               PDFCanvas.AddError("Failed to initialize color management!");
            }
         }
         // Skip anything that is not required to display a page.
         // Independent of the used flags this instance is not used to load pages! The page cache uses
         // this instance only to fetch the page's bounding boxes and orientation.
         m_PDF.SetImportFlags(TImportFlags.ifContentOnly | TImportFlags.ifImportAsPage | TImportFlags.ifAllAnnots | TImportFlags.ifFormFields);
         m_PDF.SetImportFlags2(TImportFlags2.if2UseProxy | TImportFlags2.if2NoResNameCheck);
         // Load external CMaps delayed. Note that the cache contains its own functions to load the CMaps. Loading the
         // CMaps into the main PDF instance is useless because this instance is not used to render pages...

         // In a real application make sure that you copy the files into a directory that DynaPDF can access at runtime.
         path = System.IO.Path.GetFullPath("../../../../../../../Resource/CMap/");
         CPDF.rasSetCMapDir(m_Cache, path, TLoadCMapFlags.lcmRecursive | TLoadCMapFlags.lcmDelayed);
         
         Graphics g = this.CreateGraphics();
         PDFCanvas.SetScrollLineDelta(true, Convert.ToInt32(g.DpiY) * 80 / 72);
      }

      private void BtnCollapseAll_Click(object sender, EventArgs e)
      {
         TreeBookmarks.CollapseAll();
      }

      private void BtnErrors_Click(object sender, EventArgs e)
      {
         m_ErroLog.ShowDialog();
         if (m_ErroLog.Errors.Items.Count == 0) BtnErrors.Visible = false;
      }

      private void BtnExpandAll_Click(object sender, EventArgs e)
      {
         TreeBookmarks.ExpandAll();
      }

      private void BtnHideTabControl_Click(object sender, EventArgs e)
      {
         Splitter.Visible   = false;
         TabControl.Visible = false;
      }

      private void MnuExit_Click(object sender, EventArgs e)
      {
         Application.Exit();
      }

      // Something like a radio button group was already supported in Visual Basic 6 and this version is 13 years old!!!
      private void BtnFitBest_Click(object sender, EventArgs e)
      {
         PDFCanvas.PageScale = TPDFPageScale.psFitBest;
         // Zooming can change the page layout to plSinglePage
         if (PDFCanvas.PageLayout == TPageLayout.plOneColumn)
         {
            BtnViewContinous.Checked  = true;
            BtnViewSinglePage.Checked = false;
         }
         if (BtnFitBest.Checked)
         {
            BtnFitWidth.Checked  = false;
            BtnFitHeight.Checked = false;
         }else
            BtnFitBest.Checked = true;
      }

      private void BtnFitHeight_Click(object sender, EventArgs e)
      {
         PDFCanvas.PageScale = TPDFPageScale.psFitHeight;
         if (PDFCanvas.PageLayout == TPageLayout.plOneColumn)
         {
            BtnViewContinous.Checked  = true;
            BtnViewSinglePage.Checked = false;
         }
         if (BtnFitHeight.Checked)
         {
            BtnFitWidth.Checked = false;
            BtnFitBest.Checked  = false;
         }else
            BtnFitHeight.Checked = true;
      }

      private void BtnFitWidth_Click(object sender, EventArgs e)
      {
         PDFCanvas.PageScale = TPDFPageScale.psFitWidth;
         if (PDFCanvas.PageLayout == TPageLayout.plOneColumn)
         {
            BtnViewContinous.Checked  = true;
            BtnViewSinglePage.Checked = false;
         }
         if (BtnFitWidth.Checked)
         {
            BtnFitBest.Checked   = false;
            BtnFitHeight.Checked = false;
         }else
            BtnFitWidth.Checked = true;
      }

      private void BtnRotate90_Click(object sender, EventArgs e)
      {
         PDFCanvas.Rotate = PDFCanvas.Rotate + 90;
      }

      private void BtnRotateM90_Click(object sender, EventArgs e)
      {
         PDFCanvas.Rotate = PDFCanvas.Rotate - 90;
      }

      private void BtnRotate180_Click(object sender, EventArgs e)
      {
         PDFCanvas.Rotate = PDFCanvas.Rotate + 180;
      }

      private void BtnViewContinous_Click(object sender, EventArgs e)
      {
         if (PDFCanvas.PageCount > 1)
         {
            PDFCanvas.PageLayout = TPageLayout.plOneColumn;
            // This can happen when we are in zoom mode
            if (PDFCanvas.PageLayout == TPageLayout.plSinglePage)
            {
               BtnViewContinous.Checked  = false;
               BtnViewSinglePage.Checked = true;
            }else
            {
               BtnViewContinous.Checked  = true;
               BtnViewSinglePage.Checked = false;
            }
         }else
         {
            BtnViewContinous.Checked  = false;
            BtnViewSinglePage.Checked = true;
         }
      }

      private void BtnViewSinglePage_Click(object sender, EventArgs e)
      {
         BtnViewContinous.Checked  = false;
         BtnViewSinglePage.Checked = true;
         PDFCanvas.PageLayout = TPageLayout.plSinglePage;
      }

      private void BtnZoomIn_Click(object sender, EventArgs e)
      {
         Single zoom;
         // Consider the resolutiuon of the output device
         zoom = CPDF.rasGetCurrZoom(m_Cache) * 72.0f / PDFCanvas.Resolution * 1.25f;
         if (zoom <= 0.25)
            zoom = 0.25f;
         else if (zoom <= 0.5f)
            zoom = 0.5f;
         else if (zoom <= 0.75f)
            zoom = 0.75f;
         else if (zoom <= 1.0f)
            zoom = 1.0f;
         else if (zoom <= 1.25)
            zoom = 1.25f;
         else if (zoom <= 1.5f)
            zoom = 1.5f;
         else if (zoom <= 2.0f)
            zoom = 2.0f;
         else if (zoom <= 3.0f)
            zoom = 3.0f;
         else if (zoom <= 4.0f)
            zoom = 4.0f;
         else if (zoom <= 8.0f)
            zoom = 8.0f;
         else if (zoom <= 16.0f)
            zoom = 16.0f;
         else if (zoom <= 24.0f)
            zoom = 24.0f;
         else if (zoom <= 32.0f)
            zoom = 32.0f;
         else
            zoom = 64.0f;

         PDFCanvas.Zoom(zoom * PDFCanvas.Resolution / 72.0f);
         // If the zoom factor is large then DynaPDF falls back into zoom mode and this mode works with the
         // page layout plSinglePage. The previous mode will be restored when the zoom factor becomes small
         // enough to render the pages as usual.
         if (PDFCanvas.PageLayout == TPageLayout.plSinglePage)
         {
            BtnViewContinous.Checked  = false;
            BtnViewSinglePage.Checked = true;
         }
         UnCheckZoomButtons();
      }

      private void BtnZoomOut_Click(object sender, EventArgs e)
      {
         Single zoom;
         // Consider the resolution of the ouput device
         zoom = CPDF.rasGetCurrZoom(m_Cache) * 72.0f / PDFCanvas.Resolution * 100.0f;
         // 10% tolerance to avoid rounding errors
         if (zoom > 3550.0f)
            zoom = 32.0f;
         else if (zoom > 2650.0f)
            zoom = 24.0f;
         else if (zoom > 1750.0f)
            zoom = 16.0f;
         else if (zoom > 880.0f)
            zoom = 8.0f;
         else if (zoom > 440.0f)
            zoom = 4.0f;
         else if (zoom > 330.0f)
            zoom = 3.0f;
         else if (zoom > 220.0f)
            zoom = 2.0f;
         else if (zoom > 165.0f)
            zoom = 1.5f;
         else if (zoom > 132.5f)
            zoom = 1.25f;
         else if (zoom > 110.0f)
            zoom = 1.0f;
         else if (zoom > 82.5f)
            zoom = 0.75f;
         else if (zoom > 55.0f)
            zoom = 0.5f;
         else if (zoom > 27.5f)
            zoom = 0.25f;
         else
            zoom = 0.1f;

         // The zoom factor that we have calculated is measured in 1/72 inch units.
         // We must multiply the value with the resolution of the output device divided by 72 to get the value we need.
         PDFCanvas.Zoom(zoom * PDFCanvas.Resolution / 72.0f);
         // Restore the page layout if necessary
         if (PDFCanvas.PageLayout == TPageLayout.plOneColumn)
         {
            BtnViewContinous.Checked  = true;
            BtnViewSinglePage.Checked = false;
         }
         UnCheckZoomButtons();
      }

      private void CboZoom_KeyUp(object sender, KeyEventArgs e)
      {
         if (e.KeyCode == Keys.Return)
         {
            if (CboZoom.Text.Length == 0) return;
            try
            {
               CboZoom.Text = CboZoom.Text.Trim('%');
               Single zoom = Convert.ToSingle(CboZoom.Text) / 100.0f;
               if (zoom < 0.01f)
                  zoom = 0.01f;
               else if (zoom > 64.0f)
                  zoom = 64.0f;

               PDFCanvas.Zoom(zoom * PDFCanvas.Resolution / 72.0f);
               // If the zoom factor is large then DynaPDF falls back into zoom mode and this mode works with the
               // page layout plSinglePage. The previous mode will be restored when the zoom factor becomes small
               // enough to render the pages as usual.
               if (PDFCanvas.PageLayout == TPageLayout.plSinglePage)
               {
                  BtnViewContinous.Checked  = false;
                  BtnViewSinglePage.Checked = true;
               }
               UnCheckZoomButtons();
            }catch
            {}
         }
      }

      private void CboZoom_SelectedIndexChanged(object sender, EventArgs e)
      {
         Single zoom;
         switch(CboZoom.SelectedIndex)
         {
            case 0:  zoom = 0.1f;  break;
            case 1:  zoom = 0.25f; break;
            case 2:  zoom = 0.5f;  break;
            case 3:  zoom = 0.75f; break;
            case 4:  zoom = 1.0f;  break;
            case 5:  zoom = 1.25f; break;
            case 6:  zoom = 1.5f;  break;
            case 7:  zoom = 2.0f;  break;
            case 8:  zoom = 3.0f;  break;
            case 9:  zoom = 4.0f;  break;
            case 10: zoom = 8.0f;  break;
            case 11: zoom = 16.0f; break;
            case 12: zoom = 24.0f; break;
            case 13: zoom = 32.0f; break;
            case 14: zoom = 64.0f; break;
            default: return;
         }
         // Consider the resolution of the ouput device
         PDFCanvas.Zoom(zoom * PDFCanvas.Resolution / 72.0f);
         switch(PDFCanvas.PageLayout)
         {
            case TPageLayout.plSinglePage:
            {
               BtnViewContinous.Checked  = false;
               BtnViewSinglePage.Checked = true;
               break;
            }
            case TPageLayout.plOneColumn:
            {
               BtnViewContinous.Checked  = true;
               BtnViewSinglePage.Checked = true;
               break;
            }
         }
         UnCheckZoomButtons();
      }

      private void EnableControls(bool Enable)
      {
         BtnFitBest.Enabled        = Enable;
         BtnFitHeight.Enabled      = Enable;
         BtnFitWidth.Enabled       = Enable;
         BtnRotate90.Enabled       = Enable;
         BtnRotateM90.Enabled      = Enable;
         BtnRotate180.Enabled      = Enable;
         BtnViewContinous.Enabled  = Enable;
         BtnViewSinglePage.Enabled = Enable;
         BtnZoomIn.Enabled         = Enable;
         BtnZoomOut.Enabled        = Enable;
         CboZoom.Enabled           = Enable;
         MnuView.Enabled           = Enable;
         TxtPageNum.Enabled        = Enable;
      }

      private void MnuCloseFile_Click(object sender, EventArgs e)
      {
         BtnErrors.Visible = false;
         PDFCanvas.CloseFile();
         TreeBookmarks.Nodes.Clear();
         m_Bookmarks = null;
         EnableControls(false);
         Splitter.Visible   = false;
         TabControl.Visible = false;
         TxtPageNum.Text   = "0";
         LblPageCount.Text = "of 0";
      }

      private void MnuViewDisablePageLayout_Click(object sender, EventArgs e)
      {
         if (MnuViewDisablePageLayout.Checked)
            m_OpenFlags |= TInitCacheFlags.icfIgnorePageLayout;
         else
            m_OpenFlags &= ~TInitCacheFlags.icfIgnorePageLayout;
      }

      private void MnuViewIngoreOpenAction_Click(object sender, EventArgs e)
      {
         if (MnuViewIngoreOpenAction.Checked)
            m_OpenFlags |= TInitCacheFlags.icfIgnoreOpenAction;
         else
            m_OpenFlags &= ~TInitCacheFlags.icfIgnoreOpenAction;
      }

      private void MnuOpenFile_Click(object sender, EventArgs e)
      {
         if (OpenDialog.ShowDialog() == DialogResult.OK)
         {
            OpenPDFFile(OpenDialog.FileName);
         }
      }

      private void MnuRotate0_Click(object sender, EventArgs e)
      {
         PDFCanvas.Rotate = 0;
      }

      private void MnuViewBookmarks_Click(object sender, EventArgs e)
      {
         ShowBookmarks();
         Splitter.Visible    = true;
         TabControl.Visible  = true;
         TabControl.TabIndex = 0;
      }

      private void OpenPDFFile(String FileName)
      {
         int retval;
         // Clean up if necessary
         MnuCloseFile_Click(null, null);
         m_PDF.CreateNewPDF(null);
         retval = m_PDF.OpenImportFile(FileName, TPwdType.ptOpen, null);
         if (retval < 0)
         {
            while (m_PDF.IsWrongPwd(retval))
            {
               m_PDF.ClearErrorLog(); // Remove the "Wrong password error message"
               frmPassword pwdDlg = new frmPassword();
               if (pwdDlg.ShowDialog() == DialogResult.OK)
               {
                  retval = m_PDF.OpenImportFile(FileName, TPwdType.ptOpen, pwdDlg.Password);
                  if (m_PDF.IsWrongPwd(retval))
                  {
                     m_PDF.ClearErrorLog(); // Remove the "Wrong password error message"
                     retval = m_PDF.OpenImportFile(FileName, TPwdType.ptOwner, pwdDlg.Password);
                  }
               }else
                  break;
            }
            if (retval < 0)
            {
               MnuCloseFile_Click(null, null);
               // At this point the page cache doesn't notice when an error occurs.
               // The error messages are still available after the file was already closed.
               if (m_PDF.GetErrLogMessageCount() > 0)
               {
                  TPDFError err = new TPDFError();
                  if (m_PDF.GetErrLogMessage(0, ref err))
                  {
                     PDFCanvas.AddError(err.Message);
                     BtnErrors.Visible = true;
                  }
               }
               return;
            }
         }
         // Display the outline panel if necessary before initializing the PDFCanvas.
         // This should be done here because displaying the outline panel causes a resize event.
         if (m_PDF.GetPageMode() == (int)TPageMode.pmUseOutlines)
         {
            Splitter.Visible    = true;
            TabControl.Visible  = true;
            TabControl.TabIndex = 0;
         }
         if (!PDFCanvas.InitBaseObjects(m_OpenFlags))
         {
            MnuCloseFile_Click(null, null);
            return;
         }
         if (m_PDF.GetInRepairMode())
            PDFCanvas.AddError("Opened damaged PDF file in repair mode!");

         // Process the errors of the top level objects if any
         PDFCanvas.ProcessErrors(false);
         TxtPageNum.Text   = String.Format("{0}", PDFCanvas.FirstPage);
         LblPageCount.Text = String.Format(" of {0} ", PDFCanvas.PageCount);
         ActiveForm.Text = FileName;
         EnableControls(true);
         if (PDFCanvas.PageLayout == TPageLayout.plSinglePage)
         {
            BtnViewContinous.Checked  = false;
            BtnViewSinglePage.Checked = true;
         }else
         {
            BtnViewContinous.Checked  = true;
            BtnViewSinglePage.Checked = false;
         }
         PDFCanvas.DisplayFirstPage();
         // Show the outline tree if necessary
         if (m_PDF.GetPageMode() == (int)TPageMode.pmUseOutlines)
         {
            ShowBookmarks();
         }
      }

      private void PDFCanvas_Error(object sender, PDFControl.ErrorArgs e)
      {
         if (m_ErroLog == null)
            m_ErroLog = new frmErrorLog();
         else
            m_ErroLog.Errors.Items.Clear();
         for (int i = 0; i < e.Errors.Count; i++)
            m_ErroLog.Errors.Items.Add(e.Errors[i]);
         BtnErrors.Visible = true;
      }

      private void PDFCanvas_NewPage(object sender, PDFControl.NewPageArgs e)
      {
         TxtPageNum.Text = e.PageNum.ToString();
         CboZoom.Text    = String.Format("{0:f1}%", (CPDF.rasGetCurrZoom(m_Cache) * 72.0f / PDFCanvas.Resolution) * 100.0); // Consider the resolution of the ouput device
      }

      private String RemoveControls(String Value)
      {
         char[] retval = Value.ToCharArray();
         for (int i = 0; i < Value.Length; i++)
         {
            if (Value[i] < (char)32)
               retval[i] = (char)32;
         }
         return new String(retval);
      }

      private void ShowBookmarks()
      {
         if (TreeBookmarks.Nodes.Count > 0) return;
         this.Cursor = Cursors.WaitCursor;

         int count = m_PDF.ImportBookmarks();
         // Add a dummy node so that we don't import the bookmarks multiple times
         if (count <= 0)
            TreeBookmarks.Nodes.Add(" ");
         else
         {
            try
            {
               TBmk bm;
               FontStyle style;
               TreeNode node, parent;
               TBookmark bmk = new TBookmark();
               m_Bookmarks = new ArrayList(count);
               Splitter.Visible    = true;
               TabControl.Visible  = true;
               TabControl.TabIndex = 0;
               // The treeview is relatively fast. It is of course possible to optimize the code so that only the visible nodes
               // will be loaded.
               for (int i = 0; i < count; i++)
               {
                  if (m_PDF.GetBookmark(i, ref bmk))
                  {
                     if (bmk.Parent > -1)
                     {
                        // We must use a mapping table to map bookmark indexes to the TreeView nodes and vice versa. The check
                        // whether the parent index is valid is just for safety. You can be sure that the children of a bookmark
                        // occur after the parent bookmark.
                        if (bmk.Parent < m_Bookmarks.Count)
                        {
                           bm     = (TBmk)m_Bookmarks[bmk.Parent];
                           parent = bm.Node;
                           // Bookmark titles contain often control characters especially if the file was created on Mac OS X.
                           // Acrobat doesn't display characters below 32, so, we do the same...
                           node   = parent.Nodes.Add(RemoveControls(bmk.Title));
                           style  = FontStyle.Regular;
                           if ((bmk.Style & TBmkStyle.bmsBold) == TBmkStyle.bmsBold)
                              style |= FontStyle.Bold;
                           if ((bmk.Style & TBmkStyle.bmsItalic) == TBmkStyle.bmsItalic)
                              style |= FontStyle.Italic;
                           if (style != FontStyle.Regular)
                              node.NodeFont  = new Font(TreeBookmarks.Font, style);
                           node.ForeColor = Color.FromArgb((int)bmk.Color);
                           // Check whether the node must be expandet but expand it only one time!
                           if (bmk.Open && parent.Nodes.Count == 1)
                              parent.Expand();
                           bm = new TBmk(i, node);
                           node.Tag = bm;
                           m_Bookmarks.Add(bm);
                        }
                     }else
                     {
                        node   = TreeBookmarks.Nodes.Add(RemoveControls(bmk.Title));
                        style  = FontStyle.Regular;
                        if ((bmk.Style & TBmkStyle.bmsBold) == TBmkStyle.bmsBold)
                           style |= FontStyle.Bold;
                        if ((bmk.Style & TBmkStyle.bmsItalic) == TBmkStyle.bmsItalic)
                           style |= FontStyle.Italic;
                        if (style != FontStyle.Regular)
                           node.NodeFont  = new Font(TreeBookmarks.Font, style);
                        node.ForeColor = Color.FromArgb((int)bmk.Color);
                        bm = new TBmk(i, node);
                        node.Tag = bm;
                        m_Bookmarks.Add(bm);
                     }
                  }
               }
            }catch(Exception e)
            {
               MessageBox.Show(e.Message);
            }
         }
         this.Cursor = Cursors.Default;
         // Check for errors. It is not allowed to access the error log directly with pdfGetErrLogMessage()
         // because this can cause collusions with the rendering thread!
         PDFCanvas.ProcessErrors(true);
      }

      private void TreeBookmarks_AfterSelect(object sender, TreeViewEventArgs e)
      {
         TBmk bmk = (TBmk)e.Node.Tag;
         if (bmk != null)
         {
            TUpdBmkAction retval = PDFCanvas.ExecBookmark(bmk.Index);
            if ((retval & TUpdBmkAction.ubaZoom) == TUpdBmkAction.ubaZoom)
               UnCheckZoomButtons();
            else if ((retval & TUpdBmkAction.ubaPageScale) == TUpdBmkAction.ubaPageScale)
            {
               switch(PDFCanvas.PageScale)
               {
                  case TPDFPageScale.psFitWidth:  BtnFitWidth.Checked  = true; BtnFitHeight.Checked = false; BtnFitBest.Checked   = false; break;
                  case TPDFPageScale.psFitHeight: BtnFitHeight.Checked = true; BtnFitWidth.Checked  = false; BtnFitBest.Checked   = false; break;
                  case TPDFPageScale.psFitBest:   BtnFitBest.Checked   = true; BtnFitWidth.Checked  = false; BtnFitHeight.Checked = false; break;
               }
            }
         }
      }

      private void TxtPageNum_KeyUp(object sender, KeyEventArgs e)
      {
         if (e.KeyCode == Keys.Return)
         {
            if (TxtPageNum.Text.Length == 0) return;
            try
            {
               int pageNum = Convert.ToInt32(TxtPageNum.Text);
               if (pageNum > PDFCanvas.PageCount)
                  pageNum = PDFCanvas.PageCount;
               else if (pageNum < 1)
                  pageNum = 1;

               PDFCanvas.ScrollTo(pageNum);
            }catch
            {}
         }
      }

      private void UnCheckZoomButtons()
      {
         BtnFitWidth.Checked  = false;
         BtnFitBest.Checked   = false;
         BtnFitHeight.Checked = false;
      }

      private IntPtr          m_Cache;
      private ArrayList       m_Bookmarks;
      private frmErrorLog     m_ErroLog;
      private TInitCacheFlags m_OpenFlags;
      private CPDF            m_PDF;

      private class TBmk
      {
         public int      Index;
         public TreeNode Node;
         public TBmk(int Idx, TreeNode Nde)
         {
            Index = Idx;
            Node  = Nde;
         }
      }
   }
}